//
//  PredictionEngine.swift
//  death_app Watch App
//
//  Core prediction engine that combines health data with actuarial models
//

import Foundation
import HealthKit

class PredictionEngine: ObservableObject {
    static let shared = PredictionEngine()
    
    private let baselineCalculator = BaselineLifeExpectancy()
    private let cache = PredictionCache()
    
    private init() {}
    
    /// Calculate life expectancy combining all risk factors
    func calculateLifeExpectancy(
        age: Int,
        gender: Gender,
        healthData: PredictionHealthData,
        surveyData: SurveyData
    ) -> LifeExpectancyPrediction {
        
        // Check cache first
        let cacheKey = generateCacheKey(age: age, gender: gender, healthData: healthData, surveyData: surveyData)
        if let cached = cache.getPrediction(for: cacheKey) {
            return cached
        }
        
        // Get baseline life expectancy
        let baseline = baselineCalculator.getBaseline(age: age, gender: gender)
        
        // Calculate individual risk factors
        let heartRateRisk = calculateHeartRateRisk(restingHeartRate: healthData.restingHeartRate)
        let vo2MaxRisk = calculateVO2MaxRisk(vo2Max: healthData.vo2Max, age: age, gender: gender)
        let smokingRisk = calculateSmokingRisk(smokingStatus: surveyData.smokingStatus, cigarettesPerDay: surveyData.cigarettesPerDay)
        let bmiRisk = calculateBMIRisk(bmi: healthData.bmi)
        let sleepRisk = calculateSleepRisk(averageSleepHours: healthData.averageSleepHours)
        
        // Combine risk factors using Cox proportional hazards model
        let combinedHazardRatio = combineRiskFactors([
            heartRateRisk,
            vo2MaxRisk,
            smokingRisk,
            bmiRisk,
            sleepRisk
        ])
        
        // Apply combined risk to baseline
        let adjustedLifeExpectancy = baseline * (2.0 - combinedHazardRatio)
        
        let prediction = LifeExpectancyPrediction(
            lifeExpectancy: adjustedLifeExpectancy,
            baseline: baseline,
            riskFactors: RiskFactorBreakdown(
                heartRate: heartRateRisk,
                vo2Max: vo2MaxRisk,
                smoking: smokingRisk,
                bmi: bmiRisk,
                sleep: sleepRisk
            ),
            combinedHazardRatio: combinedHazardRatio,
            calculatedDate: Date(),
            changeFromPrevious: calculatePredictionChange(newPrediction: adjustedLifeExpectancy)
        )
        
        // Cache the result
        cache.storePrediction(prediction, for: cacheKey)
        
        return prediction
    }
    
    // MARK: - Risk Factor Calculations
    
    /// Calculate heart rate risk factor (Jensen 2013)
    /// 16% increase per 10 bpm above 60
    private func calculateHeartRateRisk(restingHeartRate: Double) -> Double {
        guard restingHeartRate > 0 else { return 1.0 }
        
        let baselineRate = 60.0
        let riskPerBeat = 0.016
        
        if restingHeartRate <= baselineRate {
            return 1.0 // No additional risk
        }
        
        let excessBeats = restingHeartRate - baselineRate
        let riskIncrease = (excessBeats / 10.0) * riskPerBeat
        
        return 1.0 + riskIncrease
    }
    
    /// Calculate VO2 max risk factor (Kodama 2009)
    /// Percentile-based mortality adjustment
    private func calculateVO2MaxRisk(vo2Max: Double, age: Int, gender: Gender) -> Double {
        guard vo2Max > 0 else { return 1.2 } // Default higher risk for unknown fitness
        
        let percentile = getVO2MaxPercentile(vo2Max: vo2Max, age: age, gender: gender)
        
        // Higher percentiles = lower risk
        switch percentile {
        case 90...100: return 0.7  // Very fit, reduced risk
        case 75..<90: return 0.85  // Above average fitness
        case 50..<75: return 1.0   // Average fitness
        case 25..<50: return 1.15  // Below average fitness
        case 10..<25: return 1.3   // Poor fitness
        default: return 1.5        // Very poor fitness
        }
    }
    
    /// Calculate smoking risk factor (Doll 2004)
    /// 1.7x hazard ratio for 10 cigarettes/day
    private func calculateSmokingRisk(smokingStatus: SmokingStatus, cigarettesPerDay: Int) -> Double {
        switch smokingStatus {
        case .never:
            return 1.0
        case .former:
            // Simplified former smoker risk (would need years quit data for full calculation)
            return 1.3 // Residual risk for former smokers
        case .current:
            let baselineRisk = 1.7 // Risk for 10 cigs/day
            let additionalRisk = max(0, Double(cigarettesPerDay - 10)) * 0.05 // 5% per additional cigarette
            return baselineRisk + additionalRisk
        }
    }
    
    /// Calculate BMI risk factor (Whitlock 2009)
    private func calculateBMIRisk(bmi: Double) -> Double {
        guard bmi > 0 else { return 1.0 }
        
        switch bmi {
        case 0..<18.5: return 1.4    // Underweight
        case 18.5..<25: return 1.0   // Normal
        case 25..<30: return 1.1     // Overweight
        case 30..<35: return 1.3     // Obese I
        case 35..<40: return 1.5     // Obese II
        default: return 1.8          // Obese III
        }
    }
    
    /// Calculate sleep risk factor (Cappuccio 2010)
    private func calculateSleepRisk(averageSleepHours: Double) -> Double {
        guard averageSleepHours > 0 else { return 1.2 }
        
        let optimal = 7.5
        let deviation = abs(averageSleepHours - optimal)
        
        switch deviation {
        case 0..<0.5: return 1.0     // Optimal sleep
        case 0.5..<1.0: return 1.05  // Slightly off
        case 1.0..<1.5: return 1.1   // Moderately off
        case 1.5..<2.5: return 1.2   // Poor sleep
        default: return 1.4          // Very poor sleep
        }
    }
    
    // MARK: - Risk Combination
    
    /// Combine risk factors using Cox proportional hazards model
    private func combineRiskFactors(_ hazardRatios: [Double]) -> Double {
        // In Cox model, hazard ratios multiply
        let combinedRatio = hazardRatios.reduce(1.0, *)
        
        // Cap the combined risk to prevent unrealistic values
        return min(combinedRatio, 3.0) // Max 3x baseline risk
    }
    
    // MARK: - Change Calculation
    
    /// Calculate change from previous prediction
    private func calculatePredictionChange(newPrediction: Double) -> PredictionChange? {
        guard let lastPrediction = cache.getLastPrediction() else {
            return nil
        }
        
        let changeDays = (newPrediction - lastPrediction.lifeExpectancy) * 365.25
        let changeDate = Date()
        
        return PredictionChange(
            changeDays: changeDays,
            changeDate: changeDate,
            previousDate: lastPrediction.calculatedDate
        )
    }
    
    // MARK: - Helper Methods
    
    private func generateCacheKey(age: Int, gender: Gender, healthData: PredictionHealthData, surveyData: SurveyData) -> String {
        return "\(age)_\(gender.rawValue)_\(healthData.hashValue)_\(surveyData.hashValue)"
    }
    
    private func getVO2MaxPercentile(vo2Max: Double, age: Int, gender: Gender) -> Double {
        // Simplified percentile calculation - in real implementation would use detailed tables
        let ageGroup = age / 10 * 10 // Round to nearest decade
        
        let expectedVO2Max: Double
        if gender == .male {
            expectedVO2Max = max(20, 50 - Double(age - 20) * 0.4)
        } else {
            expectedVO2Max = max(15, 40 - Double(age - 20) * 0.3)
        }
        
        let ratio = vo2Max / expectedVO2Max
        
        // Convert ratio to rough percentile
        switch ratio {
        case 1.3...: return 95
        case 1.2..<1.3: return 85
        case 1.1..<1.2: return 75
        case 0.9..<1.1: return 50
        case 0.8..<0.9: return 25
        case 0.7..<0.8: return 15
        default: return 5
        }
    }
}